﻿///
/// アーカイブボリュームの格納アイテムの情報を表す ArchiveContent クラスの宣言を含むヘッダファイル。
///

#if       !defined(INCLUDED_SZ_ARCHIVE_CONTENT)
#define            INCLUDED_SZ_ARCHIVE_CONTENT

#if       !defined(INCLUDED_SZ_COMMON)
#include  <szCommon.hpp>
#endif // !defined(INCLUDED_SZ_COMMON)

#if       !defined(INCLUDED_SZ_FORWARD_DECLARATIONS)
#include  <szForwardDeclarations.hpp>
#endif // !defined(INCLUDED_SZ_FORWARD_DECLARATIONS)

#if       !defined(INCLUDED_SZ_ARCHIVE_PROPERTY)
#include  <szArchiveProperty.hpp>
#endif // !defined(INCLUDED_SZ_ARCHIVE_PROPERTY)

SZ_NS_BEG(szpp)

/// <summary>
/// アーカイブボリュームの格納アイテムなどの格納情報を管理するためのクラス。
/// 本クラスが管理する主要な情報は以下の二つである：
/// - 格納アイテムのフラットなリスト
/// - 格納アイテムのリストをツリーに直した仮想的なファイルシステム構造
/// 前者は主に圧縮展開時に順次処理するときと後者を構築するときに利用され、後者はフロントエンドとなる epo や シェル拡張モジュールが利用する。
/// </summary>
/// <notes>
/// アーカイブボリュームの実体が変化していなければ、一度取得した格納情報は再利用できる。たとえば、シェル拡張モジュールでは、複数の COM
/// クラスオブジェクトが同一アーカイブボリュームの格納情報を参照するが、毎度パースするのではなく、必要がない限りは以前パースした内容を再利用する。
/// <p>
/// なお、厳密に言えば、本クラスは ArchiveVolumeContent と呼ぶべきであり、あくまでも単一ボリュームの内容のみを管理する。
/// 例外的に 7-Zip のマルチボリュームについては全ボリュームの情報を保持するが、それは 7-Zip のマルチボリュームが単なるバイナリ分割であり、
/// 仮想ストリームで読み込むことで実質的には単一ボリュームとして扱われている結果である。
/// </notes>
/// <todo>
/// - 情報作成日付を管理して、アーカイブボリュームのデータが変化したときに比較して、更新の必要性を判断できるようにする。
/// </todo>
class SZ_SPEC ArchiveContent : boost::noncopyable
{
public:

  /// デフォルトコンストラクタ。
  ArchiveContent();

  /// デストラクタ。
  ~ArchiveContent();

  /// ルートコンテナのポインタを取得するメソッド。
  StoredItemContainer       *GetRoot()       { return root.get(); }

  /// ルートコンテナの const ポインタを取得するメソッド。
  const StoredItemContainer *GetRoot() const { return root.get(); }

  /// 格納アイテムリストの要素数を取得するメソッド。
  u32 GetNumberOfItems() const { return (u32)items.size(); }

  /// 格納アイテムリストの特定の要素を取得するメソッド。
  const StoredItem *GetItem(u32 index) const { return &items[index]; }

  /// 格納アイテムを格納アイテムリストに追加するメソッド。
  void AddItem(StoredItem *item);

  /// 格納アイテムリストをもとに、格納アイテムコンテナの階層構造を構築するメソッド。
  void ConstructTree(const szstring &rootName);

  /// アーカイブプロパティを取得するメソッド。
  const ArchiveProperty &GetArchiveProperty() const { return archiveProperty; }

  /// アーカイブプロパティを設定するメソッド。
  void SetArchiveProperty(const ArchiveProperty &prop) { archiveProperty = prop; }

private:

  /// ルートコンテナ。格納アイテムコンテナの階層構造はここから辿る。
  std::auto_ptr<StoredItemContainer> root;

  /// 格納アイテムリスト。アーカイブボリュームに格納されているすべてのアイテムの情報が順次収められている。
  boost::ptr_vector<StoredItem> items;

  /// アーカイブボリュームのプロパティ。正確にはアーカイブ全体のプロパティだが、ボリュームのプロパティと同一視してよい。
  ArchiveProperty archiveProperty;
};

SZ_NS_END(szpp)

#endif // !defined(INCLUDED_SZ_ARCHIVE_CONTENT)
